angular.module('DFUService', []).service('DFU', function () {
    this.vendorId= 1155;
    this.productId= 57105;
    this.connection = null;
    this.deviceInfo = null;

    this.request = {
        DETACH:     0x00, // OUT, Requests the device to leave DFU mode and enter the application.
        DNLOAD:     0x01, // OUT, Requests data transfer from Host to the device in order to load them into device internal Flash. Includes also erase commands
        UPLOAD:     0x02, // IN,  Requests data transfer from device to Host in order to load content of device internal Flash into a Host file.
        GETSTATUS:  0x03, // IN,  Requests device to send status report to the Host (including status resulting from the last request execution and the state the device will enter immediately after this request).
        CLRSTATUS:  0x04, // OUT, Requests device to clear error status and move to next step
        GETSTATE:   0x05, // IN,  Requests the device to send only the state it will enter immediately after this request.
        ABORT:      0x06  // OUT, Requests device to exit the current state/operation and enter idle state immediately.
    };

    this.status = {
        OK:                 0x00, // No error condition is present.
        errTARGET:          0x01, // File is not targeted for use by this device.
        errFILE:            0x02, // File is for this device but fails some vendor-specific verification test
        errWRITE:           0x03, // Device is unable to write memory.
        errERASE:           0x04, // Memory erase function failed.
        errCHECK_ERASED:    0x05, // Memory erase check failed.
        errPROG:            0x06, // Program memory function failed.
        errVERIFY:          0x07, // Programmed memory failed verification.
        errADDRESS:         0x08, // Cannot program memory due to received address that is out of range.
        errNOTDONE:         0x09, // Received DFU_DNLOAD with wLength = 0, but device does not think it has all of the data yet.
        errFIRMWARE:        0x0A, // Device's firmware is corrupt. It cannot return to run-time (non-DFU) operations.
        errVENDOR:          0x0B, // iString indicates a vendor-specific error.
        errUSBR:            0x0C, // Device detected unexpected USB reset signaling.
        errPOR:             0x0D, // Device detected unexpected power on reset.
        errUNKNOWN:         0x0E, // Something went wrong, but the device does not know what it was.
        errSTALLEDPKT:      0x0F  // Device stalled an unexpected request.
    };

    this.state = {
        appIDLE:                0, // Device is running its normal application.
        appDETACH:              1, // Device is running its normal application, has received the DFU_DETACH request, and is waiting for a USB reset.
        dfuIDLE:                2, // Device is operating in the DFU mode and is waiting for requests.
        dfuDNLOAD_SYNC:         3, // Device has received a block and is waiting for the host to solicit the status via DFU_GETSTATUS.
        dfuDNBUSY:              4, // Device is programming a control-write block into its nonvolatile memories.
        dfuDNLOAD_IDLE:         5, // Device is processing a download operation. Expecting DFU_DNLOAD requests.
        dfuMANIFEST_SYNC:       6, // Device has received the final block of firmware from the host and is waiting for receipt of DFU_GETSTATUS to begin the Manifestation phase; or device has completed the Manifestation phase and is waiting for receipt of DFU_GETSTATUS.
        dfuMANIFEST:            7, // Device is in the Manifestation phase. (Not all devices will be able to respond to DFU_GETSTATUS when in this state.)
        dfuMANIFEST_WAIT_RESET: 8, // Device has programmed its memories and is waiting for a USB reset or a power on reset. (Devices that must enter this state clear bitManifestationTolerant to 0.)
        dfuUPLOAD_IDLE:         9, // The device is processing an upload operation. Expecting DFU_UPLOAD requests.
        dfuERROR:               10 // An error has occurred. Awaiting the DFU_CLRSTATUS request.
    };

    this.connect = function (options) {
        var deviceInfo = options.deviceInfo;
        var successCallback = options.success || null;
        var errorCallback = options.error || null;

        if (!deviceInfo) {
            throw new Error('deviceInfo args are required.');
        }

        this.openDevice({
            deviceInfo: deviceInfo,
            success: successCallback,
            error: errorCallback
        });
    };

    this.getDevices = function (callback) {
        chrome.usb.getDevices({
            vendorId: self.vendorId,
            productId: self.productId
        }, function (devicesInfo) {
            var dfuDevicesInfo = devicesInfo.filter(function (deviceInfo) {
                return deviceInfo.productName.indexOf("STM32") !== -1;
            });
            if(callback) callback(dfuDevicesInfo);
        });
    };

    this.openDevice = function(options) {
        var self = this;
        var deviceInfo = options.deviceInfo;
        var successCallback = options.success || null;
        var errorCallback = options.error || null;

        self.deviceInfo = deviceInfo;

        if (!deviceInfo) {
            throw new Error('deviceInfo args are required.');
        }

        chrome.usb.openDevice(deviceInfo, function (connection) {
            if(chrome.runtime.lastError) {
                if (errorCallback) errorCallback(new Error(chrome.runtime.lastError.message));
                return;
            }
            self.claimInterface({
                interfaceNumber: 0,
                connection: connection,
                success: function () {
                    if (successCallback) successCallback();
                },
                error: function () {
                    if (errorCallback) errorCallback();
                }
            });
        });
    };

    this.disconnect = function () {
        var self = this;

        chrome.usb.closeDevice(self.connection, function() {
            if(chrome.runtime.lastError) {
                return;
            }
            self.connection = null;
            self.deviceInfo = null;
            console.log("Chrome DFU device disconnected.")
        });
    };

    this.claimInterface = function (options) {
        var self = this;
        var interfaceNumber = options.interfaceNumber;
        var connection = options.connection;
        var successCallback = options.success || null;
        var errorCallback = options.error || null;

        if(chrome.runtime.lastError) {
            if (errorCallback) errorCallback(new Error(chrome.runtime.lastError.message));
            return;
        }

        chrome.usb.claimInterface(connection, interfaceNumber, function claimed() {
            console.log('Claimed interface: ' + interfaceNumber);
            self.connection = connection;
            if (successCallback) successCallback();
        });
    };

    this.releaseInterface = function (interfaceNumber) {
        var self = this;

        chrome.usb.releaseInterface(this.connection, interfaceNumber, function released() {
            console.log('Released interface: ' + interfaceNumber);

            self.disconnect();
        });
    };

    this.resetDevice = function (callback) {
        chrome.usb.resetDevice(this.connection, function (result) {
            console.log('Reset Device: ' + result);

            if (callback) callback();
        });
    };

    this.getString = function (index, callback) {
        var self = this;

        chrome.usb.controlTransfer(self.connection, {
            'direction':    'in',
            'recipient':    'device',
            'requestType':  'standard',
            'request':      6,
            'value':        0x300 | index,
            'index':        0,  // specifies language
            'length':       255 // max length to retreive
        }, function (result) {
            if(chrome.runtime.lastError) {
                console.log('USB transfer failed! ' + result.resultCode);
                callback("", result.resultCode);
                return;
            }
            var view = new DataView(result.data);
            var length = view.getUint8(0);
            var descriptor = "";
            for (var i = 2; i < length; i += 2) {
                var charCode = view.getUint16(i, true);
                descriptor += String.fromCharCode(charCode);
            }
            callback(descriptor, result.resultCode);
        });
    };

    this.getInterfaceDescriptor = function (_interface, callback) {
        var self = this;

        chrome.usb.controlTransfer(this.connection, {
            'direction':    'in',
            'recipient':    'device',
            'requestType':  'standard',
            'request':      6,
            'value':        0x200,
            'index':        0,
            'length':       18 + _interface * 9
        }, function (result) {
            if(chrome.runtime.lastError) {
                console.log('USB transfer failed! ' + result.resultCode);
                callback({}, result.resultCode);
                return;
            }

            var buf = new Uint8Array(result.data, 9 + _interface * 9);
            var descriptor = {
                'bLength':            buf[0],
                'bDescriptorType':    buf[1],
                'bInterfaceNumber':   buf[2],
                'bAlternateSetting':  buf[3],
                'bNumEndpoints':      buf[4],
                'bInterfaceClass':    buf[5],
                'bInterfaceSubclass': buf[6],
                'bInterfaceProtocol': buf[7],
                'iInterface':         buf[8]
            };

            callback(descriptor, result.resultCode);
        });
    };

    this.getFlashInfo = function (_interface, callback) {
        var self = this;

        self.getInterfaceDescriptor(0, function (descriptor, resultCode) {
            if (resultCode) {
                callback({}, resultCode);
                return;
            }

            self.getString(descriptor.iInterface, function (str, resultCode) {
                if (resultCode) {
                    callback({}, resultCode);
                    return;
                }

                // F303: "@Internal Flash  /0x08000000/128*0002Kg"
                // F405: "@Internal Flash  /0x08000000/04*016Kg,01*064Kg,07*128Kg"
                // F411: "@Internal Flash  /0x08000000/04*016Kg,01*064Kg,03*128Kg"
                // split main into [location, start_addr, sectors]
                var tmp1 = str.split('/');
                if (tmp1.length != 3 || !tmp1[0].startsWith("@Internal Flash")) {
                    callback({}, -1);
                    return;
                }
                var type = tmp1[0].trim().replace('@', '');
                var start_address = parseInt(tmp1[1]);

                // split sectors into array
                var sectors = [];
                var total_size = 0;
                var tmp2 = tmp1[2].split(',');
                if (tmp2.length < 1) {
                    callback({}, -2);
                    return;
                }
                for (var i = 0; i < tmp2.length; i++) {
                    // split into [num_pages, page_size]
                    var tmp3 = tmp2[i].split('*');
                    if (tmp3.length != 2) {
                        callback({}, -3);
                        return;
                    }
                    var num_pages = parseInt(tmp3[0]);
                    var page_size = parseInt(tmp3[1]);
                    if (!page_size) {
                        callback({}, -4);
                        return;
                    }
                    var unit = tmp3[1].slice(-2, -1);
                    switch (unit) {
                        case 'M':
                            page_size *= 1024; //  fall through to K as well
                        case 'K':
                            page_size *= 1024;
                            break;
                        default:
                            callback({}, -4);
                            return;
                    }

                    sectors.push({
                        'num_pages'    : num_pages,
                        'start_address': start_address + total_size,
                        'page_size'    : page_size,
                        'total_size'   : num_pages * page_size
                    });

                    total_size += num_pages * page_size;
                }

                var flash = {
                    'type'         : type,
                    'start_address': start_address,
                    'sectors'      : sectors,
                    'total_size'   : total_size
                };

                callback(flash, resultCode);
            });
        });
    };

    this.controlTransfer = function (direction, request, value, _interface, length, data, callback) {
        var self = this;

        if (direction == 'in') {
            // data is ignored
            chrome.usb.controlTransfer(this.connection, {
                'direction':    'in',
                'recipient':    'interface',
                'requestType':  'class',
                'request':      request,
                'value':        value,
                'index':        _interface,
                'length':       length
            }, function (result) {
                if(chrome.runtime.lastError) {
                    console.log('USB transfer failed!');
                }
                if (result.resultCode) console.log(result.resultCode);

                var buf = new Uint8Array(result.data);
                callback(buf, result.resultCode);
            });
        } else {
            var arrayBuf = new ArrayBuffer(0);

            // length is ignored
            if (data) {
                arrayBuf = new ArrayBuffer(data.length);
                var arrayBufView = new Uint8Array(arrayBuf);
                arrayBufView.set(data);
            }

            chrome.usb.controlTransfer(this.connection, {
                'direction':    'out',
                'recipient':    'interface',
                'requestType':  'class',
                'request':      request,
                'value':        value,
                'index':        _interface,
                'data':         arrayBuf
            }, function (result) {
                if(chrome.runtime.lastError) {
                    console.log('USB transfer failed!');
                }
                if (result.resultCode) console.log(result.resultCode);

                callback(result);
            });
        }
    };

    // routine calling DFU_CLRSTATUS until device is in dfuIDLE state
    this.clearStatus = function (callback) {
        var self = this;

        function check_status() {
            self.controlTransfer('in', self.request.GETSTATUS, 0, 0, 6, 0, function (data) {
                if (data[4] == self.state.dfuIDLE) {
                    callback(data);
                } else {
                    var delay = data[1] | (data[2] << 8) | (data[3] << 16);

                    setTimeout(clear_status, delay);
                }
            });
        }

        function clear_status() {
            self.controlTransfer('out', self.request.CLRSTATUS, 0, 0, 0, 0, check_status);
        }

        check_status();
    };

    this.loadAddress = function (address, callback) {
        var self = this;

        self.controlTransfer('out', self.request.DNLOAD, 0, 0, 0, [0x21, address, (address >> 8), (address >> 16), (address >> 24)], function () {
            self.controlTransfer('in', self.request.GETSTATUS, 0, 0, 6, 0, function (data) {
                if (data[4] == self.state.dfuDNBUSY) {
                    var delay = data[1] | (data[2] << 8) | (data[3] << 16);

                    setTimeout(function () {
                        self.controlTransfer('in', self.request.GETSTATUS, 0, 0, 6, 0, function (data) {
                            if (data[4] == self.state.dfuDNLOAD_IDLE) {
                                callback(data);
                            } else {
                                console.log('Failed to execute address load');
                                //self.upload_procedure(99);
                            }
                        });
                    }, delay);
                } else {
                    console.log('Failed to request address load');
                    //self.upload_procedure(99);
                }
            });
        });
    };

    // first_array = usually hex_to_flash array
    // second_array = usually verify_hex array
    // result = true/false
    this.verify_flash = function (first_array, second_array) {
        for (var i = 0; i < first_array.length; i++) {
            if (first_array[i] != second_array[i]) {
                console.log('Verification failed on byte: ' + i + ' expected: 0x' + first_array[i].toString(16) + ' received: 0x' + second_array[i].toString(16));
                return false;
            }
        }

        console.log('Verification successful, matching: ' + first_array.length + ' bytes');

        return true;
    };
});