angular.module('RFFWHIDService', []).service('RFFWHID', function ($timeout, $translate) {
    this.vendorId= 1155;
    this.productId= 22338;
    this.connectionId= null;
    this.deviceInfo = null;
    this.reportSize = 63; //Device specific - 1
    this.buffer = "";
    this.RFDump = {};
    this.busy = false;
    this.busyTimeout = null;
    this.busyDelay = 100;

    this.connect = function(options) {
        var self = this;
        var deviceInfo = options.deviceInfo;
        var successCallback = options.success || null;
        var errorCallback = options.error || null;

        if (!deviceInfo) {
            throw new Error('deviceInfo args are required.');
        }

        chrome.hid.connect(deviceInfo.deviceId, function (connection) {
            if (!connection) {
                if (errorCallback) errorCallback(new Error('Unable to connect to device.'));
            } else {
                self.connectionId = connection.connectionId;
                self.deviceInfo = deviceInfo;
                if (successCallback) successCallback(connection.connectionId);
            }
        });
    };

    this.disconnect = function() {
        if (this.connectionId) {
            chrome.hid.disconnect(this.connectionId);
            console.log("Chrome RFFWHID device disconnected.")
        }
        this.connectionId = null;
        this.deviceInfo = null;
        this.buffer = [];
    };

    this.getDevices= function (callback) {
        var self = this;

        chrome.hid.getDevices({
            vendorId: self.vendorId,
            productId: self.productId
        }, function(devicesInfo) {
            var hidDevicesInfo = devicesInfo.filter(function (deviceInfo) {
                return deviceInfo.productName.indexOf("RaceFlight FC") !== -1;
            });
            if(callback) callback(hidDevicesInfo);
        });
    };

    this.hidStrToBytes = function(string) {
        var pad = this.reportSize;
        var bytes = new Uint8Array(pad);
        var contents = string;
        var i;
        for (i = 0; i < contents.length && i < bytes.length; ++i) {
            if (contents.charCodeAt(i) > 255) {
                throw new Error("I am not smart enough to decode non-ASCII data.");
            }
            bytes[i] = contents.charCodeAt(i);
        }
        for (i = contents; i < bytes.length; ++i) {
            bytes[i] = pad;
        }

        return bytes;

    };

    this.receivedDataToStr = function(data) {
        return String.fromCharCode.apply(null, new Uint8Array(data));
    };

    this.clearBuffer = function () {
        this.buffer = "";
    };

    this.send = function(options) {
        var self = this;

        options = options || {};

        var data = options.data;
        var successCallback = options.success || null;
        var errorCallback = options.error || null;

        if (!data) {
            throw new Error('data args are required.');
        }

        if (chrome.runtime.lastError) {
            if (errorCallback) errorCallback(new Error('Internal HID error'));
            return
        }

        if (!self.connectionId) {
            if (errorCallback) errorCallback(new Error('Internal HID error'));
            return
        }

        chrome.hid.send(self.connectionId, 2, data.buffer, function() {
            if (successCallback) successCallback();
        });
    };

    this.receive = function(options) {
        options = options || {};

        var self = this;
        var successCallback = options.success || null;
        var errorCallback = options.error || null;

        if (chrome.runtime.lastError) {
            if (errorCallback) errorCallback(new Error('Receive failed: Internal HID error'));
            return
        }

        if (!self.connectionId) {
            if (errorCallback) errorCallback(new Error('Receive failed: connectionId not found.'));
            return
        }

        chrome.hid.receive(self.connectionId, function (reportId, data) {
            if (successCallback) successCallback(reportId, data);
        });
    };

    this.sendAndReceive = function(options) {
        var self = this;

        options = options || {};

        var data = options.data;
        var successCallback = options.success || null;
        var errorCallback = options.error || null;

        if (!data) {
            throw new Error('data args are required.');
        }

        self.send({
            data: data,
            success: function () {
                self.receive({
                    success: function(reportId, data) {
                        if (successCallback) successCallback(reportId, data);
                    },
                    error: function (err) {
                        if (errorCallback) errorCallback(err);
                    }
                });
            },
            error: function (err) {
                if (errorCallback) errorCallback(err);
            }
        });
    };

    this.sendCommand = function (options) {
        var self = this;

        options = options || {};
        var command = options.command || null;

        if (!command) {
            throw new Error('Illegal command argument for sendCommand.');
        }

        var data = self.hidStrToBytes(command);
        var successCallback = options.success || null;
        var errorCallback = options.error || null;
        var deadlineTime = options.deadlineTime || 6000;

        self.clearBuffer();

        var deadlineTimeout = $timeout(function() {
            $timeout.cancel(deadlineTimeout);
            deadlineTimeout = null;
            self.busy = false;
            self.busyTimeout = null;
            if (errorCallback) errorCallback(new Error(command+ ' command timeout.'));
        }, deadlineTime);


        self.busy = true;

        function handler(data) {
            self.sendAndReceive({
                data: data,
                success: function (reportId, data) {
                    $timeout(function(){
                        var responseStr = self.receivedDataToStr(data);
                        if (deadlineTimeout == null) {
                            self.busy = false;
                            self.busyTimeout = null;
                            return
                        }
                        if (responseStr.indexOf("Unknown Command") !== -1) {
                            $timeout.cancel(deadlineTimeout);
                            deadlineTimeout = null;
                            self.busy = false;
                            self.busyTimeout = null;
                            if (errorCallback) errorCallback(new Error(responseStr.indexOf("\n\0")));
                        } else if (responseStr.indexOf("noargumentsfoundforargument") !== -1) {
                            $timeout.cancel(deadlineTimeout);
                            deadlineTimeout = null;
                            self.busy = false;
                            self.busyTimeout = null;
                            if (errorCallback) errorCallback(new Error(responseStr));
                        } else if (responseStr.indexOf("\n\0") !== -1) {
                            $timeout.cancel(deadlineTimeout);
                            deadlineTimeout = null;
                            self.busy = false;
                            self.busyTimeout = null;
                            self.buffer += responseStr.substr(0, responseStr.indexOf("\n\0")); // get rid of last package garbage
                            if (successCallback) successCallback();
                        } else {
                            self.buffer += responseStr;
                            handler(self.hidStrToBytes("more"));
                        }
                    }, 0)
                },
                error: function (err) {
                    if (errorCallback) errorCallback(err);
                }
            });
        }
        handler(data);
    };

    this.dump = function(options) {
        var self = this;

        if (self.busy) {
            console.log("Board is busy. Waiting");
            self.busyTimeout = $timeout(function () {
                self.dump(options);
            }, self.busyDelay);
            return
        }

        options = options || {};
        var commandArg = options.commandArg || "all";

        var allowedCommandArgs = ["all"];
        if (allowedCommandArgs.indexOf(commandArg) == -1) {
            throw new Error('Illegal command argument for dump command.');
        }

        var populateDump = options.populateDump || false;
        var successCallback = options.success || null;
        var errorCallback = options.error || null;

        var command = "dump "+commandArg;

        self.sendCommand({
            command: command,
            success: function () {
                if (populateDump) {
                    var splitBuffer = self.buffer.split('\n');
                    self.RFDump['modes'] = {};
                    self.RFDump['set'] = {};
                    self.RFDump['fl'] = {};

                    for (var j=0; j<splitBuffer.length; j++) {

                        if (splitBuffer[j].match(/modes (\w+)=(.+)=(.+)=(.+)/)) {
                            var modes = splitBuffer[j].match(/modes (\w+)=(.+)=(.+)=(.+)/);
                            if (modes && modes.length == 5) {
                                self.RFDump['modes'][modes[1]] = {
                                    'channel': modes[2],
                                    'min': modes[3],
                                    'max': modes[4]
                                };
                            } else {
                                if (errorCallback) errorCallback(new Error($translate.instant("DUMP_LOAD_MODES_ERROR")));
                                return
                            }
                        } else if(splitBuffer[j].match(/set (\w+)=(.+)/)) {
                            var set = splitBuffer[j].match(/set (\w+)=(.+)/);
                            if (set && set.length == 3) {
                                self.RFDump['set'][set[1]] = parseFloat(set[2]) || 0;
                            } else {
                                if (errorCallback) errorCallback(new Error($translate.instant("DUMP_LOAD_VARIABLES_ERROR")));
                                return
                            }
                        }  else if(splitBuffer[j].match(/#fl (\w+)=(.+)/)) {
                            var fl = splitBuffer[j].match(/#fl (\w+)=(.+)/);
                            if (fl && fl.length == 3) {
                                self.RFDump['fl'][fl[1]] = fl[2];
                            } else {
                                if (errorCallback) errorCallback(new Error($translate.instant("DUMP_LOAD_FLIGHT_LOG_DATA_ERROR")));
                                return
                            }
                        } else if(splitBuffer[j].indexOf('#vr') == 0) {
                            var vr = splitBuffer[j].match(/(\w+):([^;]+)*/g);
                            if (vr) {
                                for (var i=0; i<vr.length; i++) {
                                    var match = vr[i].match(/(\w+):(.+)/);
                                    if (match && match.length == 3) {
                                        self.RFDump[match[1]] = match[2];
                                    }
                                }
                            } else {
                                if (errorCallback) errorCallback(new Error($translate.instant("DUMP_LOAD_MAIN_INFO_DATA_ERROR")));
                                return
                            }
                        }
                    }
                }
                if (successCallback) successCallback();
            },
            error: function (err) {
                self.busy = false;
                if (errorCallback) errorCallback(err);
            }
        });
    };

    this.setVar = function(options) {
        var self = this;

        if (self.busy) {
            console.log("Board is busy. Waiting");
            self.busyTimeout = $timeout(function () {
                self.setVar(options);
            }, self.busyDelay);
            return
        }

        options = options || {};

        var varName = options.varName;
        var varValue = options.varValue;
        var successCallback = options.success || null;
        var errorCallback = options.error || null;

        if (varName == null || varValue == null) {
            throw new Error('varName and varValue args are required.');
        }

        var command = "set "+varName+"="+varValue;

        self.busy = true;

        self.sendCommand({
            command: command,
            success: function () {
                var splitBuffer = self.buffer.split('\n');
                for (var j=0; j<splitBuffer.length; j++) {
                    if (splitBuffer[j].match(/#me (\w+)=(.+)/)) {
                        if (successCallback) successCallback();
                        return
                    }
                }
                if (errorCallback) errorCallback(new Error($translate.instant("VARIABLE_NOT_SET_ERROR", {'varName': varName})));

            },
            error: function (err) {
                if (errorCallback) errorCallback(err);
            }
        });
    };

    this.setMode = function(options) {
        var self = this;

        if (self.busy) {
            console.log("Board is busy. Waiting");
            self.busyTimeout = $timeout(function () {
                self.setMode(options);
            }, self.busyDelay);
            return
        }

        options = options || {};

        var modeName = options.mode;
        var channel = options.channel;
        var min = options.min;
        var max = options.max;
        var successCallback = options.success || null;
        var errorCallback = options.error || null;

        if (modeName == null || channel == null || min == null || max == null) {
            throw new Error('modeName, channel and min and max args are required.');
        }

        var command = "modes "+modeName+"="+channel+"="+min+"="+max;

        self.sendCommand({
            command: command,
            success: function () {
                var splitBuffer = self.buffer.split('\n');

                for (var j=0; j<splitBuffer.length; j++) {
                    if (splitBuffer[j].indexOf(modeName+" set") !== -1) {
                        if (successCallback) successCallback();
                        return
                    }
                }

                if (errorCallback) errorCallback(new Error($translate.instant("MODE_NOT_SET_ERROR", {'modeName': modeName})));
            },
            error: function (err) {
                if (errorCallback) errorCallback(err);
            }
        });
    };

    this.save = function(options) {
        var self = this;

        if (self.busy) {
            console.log("Board is busy. Waiting");
            self.busyTimeout = $timeout(function () {
                self.save(options);
            }, self.busyDelay);
            return
        }

        options = options || {};

        var successCallback = options.success || null;
        var errorCallback = options.error || null;

        var command = "save";

        self.sendCommand({
            command: command,
            success: function () {
                var splitBuffer = self.buffer.split('\n');
                for (var j=0; j<splitBuffer.length; j++) {
                    if (splitBuffer[j].match(/#ms Save Complete/)) {
                        if (successCallback) successCallback();
                        return
                    }
                }

                if (errorCallback) errorCallback(new Error($translate.instant("SAVE_ERROR")));
            },
            error: function (err) {
                if (errorCallback) errorCallback(err);
            }
        });
    };

    this.eraseFlash = function(options) {
        var self = this;

        if (self.busy) {
            console.log("Board is busy. Waiting");
            self.busyTimeout = $timeout(function () {
                self.eraseFlash(options);
            }, self.busyDelay);
            return
        }

        options = options || {};

        var successCallback = options.success || null;
        var errorCallback = options.error || null;
        var deadlineTime = options.deadlineTime || 60000;

        var command = "eraseflash";

        self.sendCommand({
            command: command,
            deadlineTime: deadlineTime,
            success: function () {
                var splitBuffer = self.buffer.split('\n');
                if (splitBuffer.length > 0) {
                    if (splitBuffer[1].match(/#me Flash Erase Complete/)) {
                        if (successCallback) successCallback();
                    } else {
                        if (errorCallback) errorCallback(new Error('Erase flash failed.'));
                    }
                } else {
                    if (errorCallback) errorCallback(new Error('Erase flash failed.'));
                }
            },
            error: function (err) {
                if (errorCallback) errorCallback(err);
            }
        });
    };

    this.eraseAllFlash = function(options) {
        var self = this;

        if (self.busy) {
            console.log("Board is busy. Waiting");
            self.busyTimeout = $timeout(function () {
                self.eraseAllFlash(options);
            }, self.busyDelay);
            return
        }

        options = options || {};

        var successCallback = options.success || null;
        var errorCallback = options.error || null;
        var deadlineTime = options.deadlineTime || 60000;

        var command = "eraseallflash";

        self.sendCommand({
            command: command,
            deadlineTime: deadlineTime,
            success: function () {
                var splitBuffer = self.buffer.split('\n');
                if (splitBuffer.length > 0) {
                    if (splitBuffer[1].match(/#me Flash Erase Complete/)) {
                        if (successCallback) successCallback();
                    } else {
                        if (errorCallback) errorCallback(new Error('Erase all flash failed.'));
                    }
                } else {
                    if (errorCallback) errorCallback(new Error('Erase all flash failed.'));
                }
            },
            error: function (err) {
                if (errorCallback) errorCallback(err);
            }
        });
    };


    this.rcdata = function(options) {
        var self = this;

        if (self.busy) {
            console.log("Board is busy. Waiting");
            self.busyTimeout = $timeout(function () {
                self.rcdata(options);
            }, self.busyDelay);
            return
        }

        options = options || {};

        var successCallback = options.success || null;
        var errorCallback = options.error || null;

        var command = "rcdata";

        self.sendCommand({
            command: command,
            success: function () {
                var splitBuffer = self.buffer.split('\n');
                var rcdata = {};
                for (var j=0; j<splitBuffer.length; j++) {
                    if (splitBuffer[j].match(/#rc (\d+)=(.+)/)) {
                        var rb = splitBuffer[j].match(/#rc (\d+)=(.+)/);
                        if (rb && rb.length == 3) {
                            rcdata[rb[1]] = {
                                'channel': rb[1],
                                'val': rb[2]
                            };
                        } else {
                            console.log('Incorrect regex for: '+splitBuffer[j]);
                        }
                    }
                }
                if (successCallback) successCallback(rcdata);
            },
            error: function (err) {
                if (errorCallback) errorCallback(err);
            }
        });
    };

    this.rxrcdata = function(options) {
        var self = this;

        if (self.busy) {
            console.log("Board is busy. Waiting");
            self.busyTimeout = $timeout(function () {
                self.rxrcdata(options);
            }, self.busyDelay);
            return
        }

        options = options || {};

        var successCallback = options.success || null;
        var errorCallback = options.error || null;

        var command = "rxrcdata";

        self.sendCommand({
            command: command,
            success: function () {
                var splitBuffer = self.buffer.split('\n');
                var rxrcdata = {};
                for (var j=0; j<splitBuffer.length; j++) {
                    if (splitBuffer[j].match(/#rb (\d+)=(\d+):(\d+)/)) {
                        var rb = splitBuffer[j].match(/#rb (\d+)=(\d+):(\d+)/);
                        if (rb && rb.length == 4) {
                            rxrcdata[rb[1]] = {
                                'channel': rb[1],
                                'min': rb[2],
                                'max': rb[3]
                            };
                        } else {
                            console.log('Incorrect regex for: '+splitBuffer[j]);
                        }
                    }
                }

                if (successCallback) successCallback(rxrcdata);
            },
            error: function (err) {
                if (errorCallback) errorCallback(err);
            }
        });
    };


    this.wizCala = function(options) {
        var self = this;

        if (self.busy) {
            console.log("Board is busy. Waiting");
            self.busyTimeout = $timeout(function () {
                self.wizCala(options);
            }, self.busyDelay);
            return
        }

        options = options || {};

        var successCallback = options.success || null;
        var errorCallback = options.error || null;

        var command = "wiz cala";

        self.sendCommand({
            command: command,
            success: function () {
                var splitBuffer = self.buffer.split('\n');
                for (var j=0; j<splitBuffer.length; j++) {
                    if (splitBuffer[j].match(/calibrationfailed/)) {
                        if (errorCallback) errorCallback(new CommandError(splitBuffer[j]));
                        return
                    }
                }
                if (successCallback) successCallback(splitBuffer);
            },
            error: function (err) {
                if (errorCallback) errorCallback(err);
            }
        });
    };

    this.wizCalb = function(options) {
        var self = this;

        if (self.busy) {
            console.log("Board is busy. Waiting");
            self.busyTimeout = $timeout(function () {
                self.wizCalb(options);
            }, self.busyDelay);
            return
        }

        options = options || {};

        var successCallback = options.success || null;
        var errorCallback = options.error || null;

        var command = "wiz calb";

        self.sendCommand({
            command: command,
            success: function () {
                var splitBuffer = self.buffer.split('\n');
                for (var j=0; j<splitBuffer.length; j++) {
                    if (splitBuffer[j].match(/calibrationfailed/)) {
                        if (errorCallback) errorCallback(new CommandError(splitBuffer[j]));
                        return
                    }
                }
                if (successCallback) successCallback(splitBuffer);
            },
            error: function (err) {
                if (errorCallback) errorCallback(err);
            }
        });
    };

    this.wizRx = function(options) {
        var self = this;

        if (self.busy) {
            console.log("Board is busy. Waiting");
            self.busyTimeout = $timeout(function () {
                self.wizRx(options);
            }, self.busyDelay);
            return
        }

        options = options || {};

        var successCallback = options.success || null;
        var errorCallback = options.error || null;

        var command = "wiz rx";

        self.sendCommand({
            command: command,
            success: function () {
                var splitBuffer = self.buffer.split('\n');
                if (successCallback) successCallback(splitBuffer);
            },
            error: function (err) {
                if (errorCallback) errorCallback(err);
            }
        });
    };

    this.wizRc = function(options) {
        var self = this;

        if (self.busy) {
            console.log("Board is busy. Waiting");
            self.busyTimeout = $timeout(function () {
                self.wizRc(options);
            }, self.busyDelay);
            return
        }


        options = options || {};

        var index = options.index;
        var successCallback = options.success || null;
        var errorCallback = options.error || null;

        if (index == null) {
            throw new Error('index argument for wiz rc.');
        }

        var command = "wiz rc"+index;

        self.sendCommand({
            command: command,
            success: function () {
                var splitBuffer = self.buffer.split('\n');
                if (successCallback) successCallback(splitBuffer);
            },
            error: function (err) {
                if (errorCallback) errorCallback(err);
            }
        });
    };

    this.rebootRFBL = function(options) {
        var self = this;

        if (self.busy) {
            console.log("Board is busy. Waiting");
            self.busyTimeout = $timeout(function () {
                self.rebootRFBL(options);
            }, self.busyDelay);
            return
        }

        options = options || {};

        var successCallback = options.success || null;
        var errorCallback = options.error || null;

        self.clearBuffer();

        var command = "rebootrfbl";
        var data = self.hidStrToBytes(command);

        self.busy = true;

        self.send({
            data: data,
            success: function () {
                self.busy = false;
                if (successCallback) successCallback();
            },
            error: function (err) {
                self.busy = false;
                if (errorCallback) errorCallback(err);
            }
        });
    };

    this.resetDFU = function(options) {
        var self = this;

        options = options || {};

        var successCallback = options.success || null;
        var errorCallback = options.error || null;

        self.clearBuffer();

        var command = "resetDFU";
        var data = self.hidStrToBytes(command);

        self.send({
            data: data,
            success: function () {
                if (successCallback) successCallback();
            },
            error: function (err) {
                if (errorCallback) errorCallback(err);
            }
        });
    };


    this.bind = function(options) {
        var self = this;

        options = options || {};

        var index = options.index;
        var successCallback = options.success || null;
        var errorCallback = options.error || null;

        if (index == null) {
            throw new Error('index argument is required for bind.');
        }

        var command = "bind"+index;

        self.sendCommand({
            command: command,
            success: function () {
                var splitBuffer = self.buffer.split('\n');
                for (var j=0; j<splitBuffer.length; j++) {
                    if (splitBuffer[j].match(/#me Binding/)) {
                        if (successCallback) successCallback();
                        return
                    }
                }
                if (errorCallback) errorCallback(new Error('Binding not set.'));
            },
            error: function (err) {
                if (errorCallback) errorCallback(err);
            }
        });
    };


    this.wizMot = function(options) {
        var self = this;

        if (self.busy) {
            console.log("Board is busy. Waiting");
            self.busyTimeout = $timeout(function () {
                self.wizMot(options);
            }, self.busyDelay);
            return
        }

        options = options || {};

        var index = options.index;
        var successCallback = options.success || null;
        var errorCallback = options.error || null;

        if (index == null) {
            throw new Error('index argument for wiz mot is required.');
        }

        var command = "wiz mot"+index;

        self.sendCommand({
            command: command,
            success: function () {
                var splitBuffer = self.buffer.split('\n');
                for (var j=0; j<splitBuffer.length; j++) {
                    if (splitBuffer[j].match(/Motor calibration failed/)) {
                        if (errorCallback) errorCallback(
                            new CommandError($translate.instant("MOTOR_CALIBRATION_DRONE_IMMOVABLE_ERROR"))
                        );
                        return
                    }
                }

                if (successCallback) successCallback(splitBuffer);
            },
            error: function (err) {
                if (errorCallback) errorCallback(err);
            }
        });
    };


    this.onewire = function(options) {
        var self = this;

        if (self.busy) {
            console.log("Board is busy. Waiting");
            self.busyTimeout = $timeout(function () {
                self.onewire(options);
            }, self.busyDelay);
            return
        }

        options = options || {};

        var commandArg = options.commandArg;

        var allowedCommandArgs = ["auto", "read"];
        if (allowedCommandArgs.indexOf(commandArg) == -1) {
            throw new Error('Illegal command argument for 1wire command.');
        }
        var successCallback = options.success || null;
        var errorCallback = options.error || null;
        var deadlineTime = options.deadlineTime || 60000;

        var command = "1wire "+commandArg;

        //var FAAAAAAAKE = "Reading ESCs...\r\nMotor 0: 16.65, #J_H_15#        #BLHELI$EFM8B21##16.65_Tones  \r\nMotor 1: 16.65, #J_H_15#        #BLHELI$EFM8B21##16.65_Tones  \r\nMotor 2: 16.65, #J_H_15#        #BLHELI$EFM8B21##16.65_Tones  \r\nMotor 3: 16.65, #J_H_15#        #BLHELI$EFM8B21##16.65_Tones  \r\nMotor 4 Disabled\r\nMotor 5 Disabled\r\nMotor 6 Disabled\r\nMotor 7 Disabled\r\n1wiredumpstart\r\nm0=beacondelay=2min\r\nm0=beaconstrength=80\r\nm0=beepstrength=91\r\nm0=brakeonstop=enable\r\nm0=demag=high\r\nm0=direction=normal\r\nm0=txprogramming=enable\r\nm0=frequency=NOT FOUND\r\nm0=maxthrottle=1880\r\nm0=minthrottle=1012\r\nm0=midthrottle=1436\r\nm0=startuppower=0.125\r\nm0=timing=medium\r\nm0=tempprotection=disable\r\nm1=beacondelay=5min\r\nm1=beaconstrength=80\r\nm1=beepstrength=91\r\nm1=brakeonstop=enable\r\nm1=demag=high\r\nm1=direction=normal\r\nm1=txprogramming=enable\r\nm1=frequency=NOT FOUND\r\nm1=maxthrottle=1880\r\nm1=minthrottle=1012\r\nm1=midthrottle=1436\r\nm1=startuppower=0.125\r\nm1=timing=medium\r\nm1=tempprotection=disable\r\nm2=beacondelay=2min\r\nm2=beaconstrength=80\r\nm2=beepstrength=91\r\nm2=brakeonstop=enable\r\nm2=demag=high\r\nm2=direction=normal\r\nm2=txprogramming=enable\r\nm2=frequency=NOT FOUND\r\nm2=maxthrottle=1876\r\nm2=minthrottle=1012\r\nm2=midthrottle=1436\r\nm2=startuppower=0.125\r\nm2=timing=medium\r\nm2=tempprotection=disable\r\nm3=beacondelay=2min\r\nm3=beaconstrength=80\r\nm3=beepstrength=91\r\nm3=brakeonstop=enable\r\nm3=demag=high\r\nm3=direction=normal\r\nm3=txprogramming=enable\r\nm3=frequency=NOT FOUND\r\nm3=maxthrottle=1876\r\nm3=minthrottle=1012\r\nm3=midthrottle=1436\r\nm3=startuppower=0.125\r\nm3=timing=medium\r\nm3=tempprotection=disable\r\n1wiredumpcomplete";

        self.sendCommand({
            command: command,
            deadlineTime: deadlineTime,
            success: function () {
                var j;
                var splitBuffer = self.buffer.split('\n');

                //splitBuffer = FAAAAAAAKE.split('\n');

                for (j=0; j<splitBuffer.length; j++) {
                    if (splitBuffer[j].match(/No ESCs detected/)) {
                        if (errorCallback) errorCallback(new CommandError($translate.instant("NO_ESC_DETECTED_ERROR")));
                        return
                    }
                }

                if (commandArg == "read") {
                    var onewireReadData = {};
                    for (j=0; j<splitBuffer.length; j++) {
                        if (splitBuffer[j].match(/(.+)=(.+)=(.+)/)) {
                            var wire = splitBuffer[j].match(/(.+)=(.+)=(.+)/);

                            if (wire && wire.length == 4) {
                                if (onewireReadData.hasOwnProperty(wire[1])) {
                                    onewireReadData[wire[1]][wire[2]] = wire[3]
                                } else {
                                    onewireReadData[wire[1]] = {}
                                }
                            } else {
                                console.log('Incorrect regex for: '+splitBuffer[j]);
                            }
                        }
                    }

                    if (successCallback) successCallback(onewireReadData);

                } else if (commandArg == "auto") {
                    if (successCallback) successCallback(splitBuffer);
                }
            },
            error: function (err) {
                if (errorCallback) errorCallback(err);
            }
        });
    };


    this.setonewireVar = function(options) {
        var self = this;

        options = options || {};

        var motor = options.motor;
        var varName = options.varName;
        var varVal = options.varVal;
        var successCallback = options.success || null;
        var errorCallback = options.error || null;

        if (motor == null || varName == null || varVal == null) {
            throw new Error('motor, varName and varVal args are required.');
        }

        var command = "1wire "+motor+"="+varName+"="+varVal;

        self.sendCommand({
            command: command,
            success: function () {
                var splitBuffer = self.buffer.split('\n');
                for (var j=0; j<splitBuffer.length; j++) {
                    if (splitBuffer[j].indexOf("Value Set!") !== -1) {
                        if (successCallback) successCallback();
                        return
                    }
                }

                if (errorCallback) errorCallback(new Error('Can\'t set mode '+'1wire '+motor+'='+varName+'='+varVal));
            },
            error: function (err) {
                //if (successCallback) successCallback();
                if (errorCallback) errorCallback(err);
            }
        });
    };


    this.onewireSave = function(options) {
        var self = this;

        options = options || {};

        var successCallback = options.success || null;
        var errorCallback = options.error || null;

        var command = "1wire save";

        self.sendCommand({
            command: command,
            success: function () {
                var splitBuffer = self.buffer.split('\n');
                //console.log(splitBuffer)
                //splitBuffer = ["Success"];

                for (var j=0; j<splitBuffer.length; j++) {
                    if (splitBuffer[j].match(/Error Saving/)) {
                        if (errorCallback) errorCallback(new CommandError(splitBuffer[j]));
                        return
                    }
                }
                if (successCallback) successCallback();

            },
            error: function (err) {
                if (errorCallback) errorCallback(err);
            }
        });
    };

    this.onewireStop = function(options) {
        var self = this;

        options = options || {};

        var successCallback = options.success || null;
        var errorCallback = options.error || null;

        var command = "1wire stop";

        // if (successCallback) successCallback();
        // return
        self.sendCommand({
            command: command,
            success: function () {
                var splitBuffer = self.buffer.split('\n');
                for (var j=0; j<splitBuffer.length; j++) {
                    if (splitBuffer[j].match(/1Wire Session Ended./)) {
                        if (successCallback) successCallback();
                        return
                    }
                }

                if (errorCallback) errorCallback(new Error('Stop 1wire failed.'));
            },
            error: function (err) {
                if (errorCallback) errorCallback(err);
            }
        });
    };


    this.idle = function(options) {
        var self = this;

        if (self.busy) {
            console.log("Board is busy. Waiting for idle command.");
            self.busyTimeout = $timeout(function () {
                self.idle(options);
            }, self.busyDelay);
            return
        }

        options = options || {};

        var index = options.index;
        var successCallback = options.success || null;
        var errorCallback = options.error || null;

        if (index == null) {
            throw new Error('index argument is required for idle command.');
        }

        var command = "Idle "+index;

        self.sendCommand({
            command: command,
            success: function () {
                var splitBuffer = self.buffer.split('\n');
                for (var j=0; j<splitBuffer.length; j++) {
                    if (splitBuffer[j].indexOf("#me Spinning Motor "+index) !== -1) {
                        if (successCallback) successCallback();
                        return
                    }
                }
                if (errorCallback) errorCallback(new Error('Spinning motor '+index+'failed.'));
            },
            error: function (err) {
                if (errorCallback) errorCallback(err);
            }
        });
    };

    this.idlestop = function(options) {
        var self = this;

        if (self.busy) {
            console.log("Board is busy. Waiting for idlestop command.");
            self.busyTimeout = $timeout(function () {
                self.idlestop(options);
            }, self.busyDelay);
            return
        }

        options = options || {};

        var successCallback = options.success || null;
        var errorCallback = options.error || null;

        var command = "idlestop";

        self.sendCommand({
            command: command,
            success: function () {
                var splitBuffer = self.buffer.split('\n');
                for (var j=0; j<splitBuffer.length; j++) {
                    if (splitBuffer[j].match(/idlestop/)) {
                        if (successCallback) successCallback();
                        return
                    }
                }
                if (errorCallback) errorCallback(new Error('Stop motors failed.'));
            },
            error: function (err) {
                if (errorCallback) errorCallback(err);
            }
        });
    };


    this.telem = function(options) {
        var self = this;

        if (self.busy) {
            console.log("Board is busy. Waiting");
            self.busyTimeout = $timeout(function () {
                self.telem(options);
            }, self.busyDelay);
            return
        }

        options = options || {};

        var successCallback = options.success || null;
        var errorCallback = options.error || null;

        var command = "telem";

        self.sendCommand({
            command: command,
            success: function () {
                var splitBuffer = self.buffer.split('\n');
                var telemdata = {};
                for (var j=0; j<splitBuffer.length; j++) {
                    if (splitBuffer[j].match(/#tm (\w+)=(.+)/)) {
                        var tm = splitBuffer[j].match(/#tm (\w+)=(.+)/);

                        if (tm && tm.length == 3) {
                            telemdata[tm[1]] = parseFloat(tm[2]) || 0.0;
                        } else {
                            console.log('Incorrect regex for: '+splitBuffer[j]);
                        }
                    }
                }
                if (successCallback) successCallback(telemdata);
            },
            error: function (err) {
                if (errorCallback) errorCallback(err);
            }
        });
    };


    this.vtxinfo = function(options) {
        var self = this;

        if (self.busy) {
            console.log("Board is busy. Waiting");
            self.busyTimeout = $timeout(function () {
                self.vtxinfo(options);
            }, self.busyDelay);
            return
        }

        options = options || {};

        var successCallback = options.success || null;
        var errorCallback = options.error || null;

        var command = "vtxinfo";

        self.sendCommand({
            command: command,
            success: function () {
                var splitBuffer = self.buffer.split('\n');
                var vtxinfodata = {};
                for (var j=0; j<splitBuffer.length; j++) {
                    if (splitBuffer[j].match(/#me (.+): (.+)/)) {
                        var vtxinfo = splitBuffer[j].match(/#me (.+): (.+)/);

                        if (vtxinfo && vtxinfo.length == 3) {
                            vtxinfodata[vtxinfo[1]] = parseFloat(vtxinfo[2]) || 0.0;
                        } else {
                            console.log('Incorrect regex for: '+splitBuffer[j]);
                        }
                    }
                }
                if (successCallback) successCallback(vtxinfodata);
            },
            error: function (err) {
                if (errorCallback) errorCallback(err);
            }
        });
    };

    this.vtxpit = function(options) {
        var self = this;

        if (self.busy) {
            console.log("Board is busy. Waiting");
            self.busyTimeout = $timeout(function () {
                self.vtxpit(options);
            }, self.busyDelay);
            return
        }

        options = options || {};

        var successCallback = options.success || null;
        var errorCallback = options.error || null;

        var command = "vtxpit";

        self.sendCommand({
            command: command,
            success: function () {
                var splitBuffer = self.buffer.split('\n');
                var vtxpitdata = {};
                for (var j=0; j<splitBuffer.length; j++) {
                    if (splitBuffer[j].match(/#me (.+): (.+)/)) {
                        var vtxpit = splitBuffer[j].match(/#me (.+): (.+)/);

                        if (vtxpit && vtxpit.length == 3) {
                            vtxpitdata[vtxpit[1]] = parseFloat(vtxpit[2]) || 0.0;
                        } else {
                            console.log('Incorrect regex for: '+splitBuffer[j]);
                        }
                    } else if (splitBuffer[j].indexOf("#me Error putting VTX into pit mode") !== -1) {
                        if (errorCallback) errorCallback(
                            new CommandError($translate.instant("VTX_MODE_ERROR"))
                        );
                        return
                    }
                }
                if (successCallback) successCallback(vtxpitdata);
            },
            error: function (err) {
                if (errorCallback) errorCallback(err);
            }
        });
    };

    this.vtxon = function(options) {
        var self = this;

        if (self.busy) {
            console.log("Board is busy. Waiting");
            self.busyTimeout = $timeout(function () {
                self.vtxon(options);
            }, self.busyDelay);
            return
        }

        options = options || {};

        var successCallback = options.success || null;
        var errorCallback = options.error || null;

        var command = "vtxon";

        self.sendCommand({
            command: command,
            success: function () {
                var splitBuffer = self.buffer.split('\n');
                var vtxondata = {};
                for (var j=0; j<splitBuffer.length; j++) {
                    if (splitBuffer[j].match(/#me (.+): (.+)/)) {
                        var vtxon = splitBuffer[j].match(/#me (.+): (.+)/);

                        if (vtxon && vtxon.length == 3) {
                            vtxondata[vtxon[1]] = parseFloat(vtxon[2]) || 0.0;
                        } else {
                            console.log('Incorrect regex for: '+splitBuffer[j]);
                        }
                    } else if (splitBuffer[j].indexOf("#me Error putting VTX into pit mode") !== -1) {
                        if (errorCallback) errorCallback(
                            new CommandError($translate.instant("VTX_MODE_ERROR"))
                        );
                        return
                    }
                }
                if (successCallback) successCallback(vtxondata);
            },
            error: function (err) {
                if (errorCallback) errorCallback(err);
            }
        });
    };


    this.dpmode = function(options) {
        var self = this;

        if (self.busy) {
            console.log("Board is busy. Waiting");
            self.busyTimeout = $timeout(function () {
                self.dpmode(options);
            }, self.busyDelay);
            return
        }

        options = options || {};

        var successCallback = options.success || null;
        var errorCallback = options.error || null;

        var command = "dpmode";

        self.sendCommand({
            command: command,
            success: function () {
                var splitBuffer = self.buffer.split('\n');
                for (var j=0; j<splitBuffer.length; j++) {
                    if (splitBuffer[j].match(/#ms Save Complete/)) {
                        if (successCallback) successCallback();
                        return
                    }
                }

                if (errorCallback) errorCallback(new Error('Enable Digital Precision Mode failed'));
            },
            error: function (err) {
                if (errorCallback) errorCallback(err);
            }
        });
    };


});